/************************************************************************
 * \file: trace_io.c
 *
 * \version: $Id: trace_io.c,v 1.16 2012/05/28 10:27:32 jayanth.mc Exp $
 *
 * This file implements the IO handling for Trace.
 *
 * \component: Gen2 Trace
 *
 * \author B. Das         bhaskar.das(o)in.bosch.com
 *         Sakthivelu S.  sakthivelu.s(o)in.bosch.com
 *\modified: Arun V
 * \copyright: (c) 2003 - 2009 ADIT
 *
 ***********************************************************************/

#include <errno.h>
 #include <termios.h>

#include "trace_base.h"

/*SWGIII-5128:
  Flag to check the connection established and send the print error message  only once in case of error  */
static BOOL  connect_estab = TRUE;

/* Forward declarations */
LOCAL ER TRACE_UART_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_UART_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_UART_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_UART_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_TCPIP_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_TCPIP_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_TCPIP_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_TCPIP_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_TCPIP_rdext(TRACE_mgr* mgr, U8* buf, S32 rdsz, S32* asiz);
LOCAL ER TRACE_SIM_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_SIM_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_SIM_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_SIM_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_dev_null(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_ioext_null(TRACE_mgr* mgr, U8* buf, S32 rdsz, S32* asiz);
LOCAL ER TRACE_wrt_epilog(TRACE_mgr* mgr);
LOCAL ER TRACE_rdt_epilog(TRACE_mgr* mgr);
LOCAL ER TRACE_tcpip_rdt_epilog(TRACE_mgr* mgr);
/* Trace IO function pointers */
const EXPORT TRACE_io_fp g_TRACE_fnctbl =
  {{TRACE_dev_null, TRACE_UART_opn,  TRACE_dev_null,  TRACE_PROXY_opn,
    TRACE_SIM_opn,  TRACE_dev_null,  TRACE_TCPIP_opn, TRACE_USBH_opn},
   {TRACE_dev_null, TRACE_UART_cls,  TRACE_dev_null,  TRACE_PROXY_cls,
    TRACE_SIM_cls,  TRACE_dev_null,  TRACE_TCPIP_cls, TRACE_USBH_cls},
   {TRACE_dev_null, TRACE_UART_rd,   TRACE_dev_null,   TRACE_PROXY_rd,
    TRACE_SIM_rd,   TRACE_dev_null,  TRACE_TCPIP_rd,  TRACE_USBH_rd},
   {TRACE_dev_null, TRACE_UART_wr,   TRACE_dev_null,   TRACE_PROXY_wr,
    TRACE_SIM_wr,   TRACE_dev_null,  TRACE_TCPIP_wr,  TRACE_USBH_wr },
   {TRACE_ioext_null, TRACE_ioext_null,  TRACE_ioext_null,  TRACE_ioext_null,
    TRACE_ioext_null, TRACE_ioext_null,  TRACE_TCPIP_rdext,  TRACE_ioext_null },
   TRACE_wrt_epilog,
   TRACE_rdt_epilog
  };

/**
 * Raise communication error
 *
 * \param mgr Pointer to trace configuration
 *
 * \return none
 */
EXPORT void TRACE_raise_comm_err(TRACE_mgr* mgr)
{
  TRACE_status_msg msg;
  U8 len                = sizeof(msg.status);
  
  memset(&msg, 0, sizeof(TRACE_status_msg));
  
  if(mgr->sh->trc.buf.blck_mode == TRACE_BLOCK_MODE)
  {
    char errMsg[TRACE_ERR_MEM_BLK_MODE_STR_SIZE]={0};
   /* temporarily revert back to no-block mode
    * so that applications won't get blocked
    */
    mgr->sh->trc.buf.blck_mode = TRACE_NO_BLOCK_MODE;
    
    sprintf( errMsg,"[raise_comm] BLOCKMODE { TRACE: %d  PROXY : %d }",(TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)? TRUE : FALSE,(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE);
    TRACE_errmem_s32IOWrite(errMsg, strlen(errMsg));
    /* send the status to TTFis */
    msg.status = (U8)TRACE_COMM_ERR;
    msg.payld[TRACE_INDEX_0] = (U8)TRACE_COMM_ERR_SET;
    (void)TRACE_snd_status(mgr,(U8*)(VP)&msg, ++len);
  }
}

/**
 * Clear communication error
 *
 * \param mgr Pointer to trace configuration
 *
 * \return none
 */
EXPORT void TRACE_clear_comm_err(TRACE_mgr* mgr)
{
  TRACE_status_msg msg;
  U8 len                = sizeof(msg.status);
  
  memset(&msg, 0, sizeof(TRACE_status_msg));
  
  if((mgr->sh->trc.d_cnfg.flgs.blck_mode != mgr->sh->trc.buf.blck_mode) &&
    (mgr->sh->trc.d_cnfg.flgs.blck_mode == TRACE_BLOCK_MODE))
  {
    char errMsg[TRACE_ERR_MEM_BLK_MODE_STR_SIZE]={0};
    /* buf mode had been changed in raise comm err
     * revert back to orginal mode that was set
     */
    mgr->sh->trc.buf.blck_mode =(TRACE_Q_BLOCK_STAT)mgr->sh->trc.d_cnfg.flgs.blck_mode;

    sprintf( errMsg,"[clear_comm] BLOCKMODE { TRACE: %d  PROXY : %d }",(TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)? TRUE : FALSE,(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE);
    TRACE_errmem_s32IOWrite(errMsg, strlen(errMsg));

    /* send the status to TTFis */
    msg.status = (U8)TRACE_COMM_ERR;
    msg.payld[TRACE_INDEX_0] = (U8)TRACE_COMM_ERR_RESET;
    (void)TRACE_snd_status(mgr,(U8*)(VP)&msg, ++len);
  }
}

LOCAL ER TRACE_tcpip_wrt_epilog(TRACE_mgr* mgr);
LOCAL ER TRACE_tcpip_wrt_epilog(TRACE_mgr* mgr)
{

  TRACE_flagAttr   Attr = {TRACE_TCPCIP_WAI_4_RDT_SIG, TWF_ORW | TWF_BITCLR,
                                                        0, TMO_FEVR};

  ER rc = TRACE_wait_flag(&mgr->sh->evt_id[EVT_TRC_Q], &Attr);

  if(rc == E_OK)
  {
    TRACE_sleep(TRACE_TCPIP_WRT_WAI);
  }
  return rc;
}

/**
 * Open TCPIP socket
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_TCPIP_opn(TRACE_mgr* mgr, TRACE_q* curq)
{
  TRACE_TCPIP_ER  rc    = E_OK;
  TRACE_io_cnfg*  cnfg  = &mgr->io_cfg[TRACE_TCPIP];
  S32             iopt  = 1;
  S32             prctl = 0;
  S32             stage = 0;

  while((stage < 2) && (rc >= E_OK))
  {
    switch(stage)
    {
      case 0:
        rc = socket(AF_INET, SOCK_STREAM, prctl);
        break;

      case 1:
        cnfg->hdev.sock_id = (TRACE_LONG)rc;
        /* set timeout for Select as zero */
        cnfg->io_timeout   = 0; 
        rc = setsockopt(cnfg->hdev.sock_id,
                           SOL_SOCKET,
                           SO_REUSEADDR,
                           &iopt,
                           sizeof(S32));
        break;

      default:
          break;
    }
    stage++;
  }
  if(rc > E_OK)
  {
    rc = E_OK;
  }
  if(rc < E_OK)
  {
    TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Socket Error happened\n");/*agv2kor*/
  }
  curq = curq; /* To avoid cimpiler warnings */
  return rc;
}

/**
 * Close TCPIP socket
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_TCPIP_cls(TRACE_mgr* mgr, TRACE_q* curq)
{
  TRACE_io_cnfg*  cnfg = &mgr->io_cfg[TRACE_TCPIP];
  TRACE_TCPIP_ER  rc   = E_OK;

  if(cnfg->hdev.sock_id > 0)
  {
    rc = close(cnfg->hdev.sock_id);
    if(rc >= E_OK)
    {
      cnfg->hdev.sock_id = 0;
      rc = E_OK;
    }
  }
  curq = curq;
  return rc;
}

/**
 * Trace write task epilog functionality.
 *
 * \parm mgr  Pointer to Trace manager
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_wrt_epilog(TRACE_mgr* mgr)
{
  ER rc = E_OK;

  switch(mgr->sh->trc.d_cnfg.actv_chan)
  {
    case TRACE_TCPIP:
      rc = TRACE_tcpip_wrt_epilog(mgr);
      break;

    case TRACE_PROXY:
    case TRACE_SIM:
    case TRACE_USBH:
    case TRACE_USBF:
    case TRACE_UART:
    case TRACE_NULLDEV:
    default:
      break;
  }

  if((rc == E_OK) && (TRUE == mgr->ctrl.snd_status))
  {
    rc = TRACE_snd_blkmode_stat(mgr); 
  }
  return rc;
}

/**
 * Establish connection with host.
 *
 * \parm mgr  Pointer to Trace manager
 *
 * return \li E_OK     If successful
 *        \li E_IO     In case of socket errors
 *        \li E_DISWAI In case of shutdown request
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_estab_connect(TRACE_mgr* mgr);
LOCAL ER TRACE_estab_connect(TRACE_mgr* mgr)
{
  TRACE_io_cnfg* cnfg          = &mgr->io_cfg[TRACE_TCPIP];
  TRACE_TCPIP_ER rc            = 0;
  S32 flags                    = 0;
  struct sockaddr_in sa_kaaddr;
  
  memset(&sa_kaaddr, 0, sizeof(sa_kaaddr));

  sa_kaaddr.sin_family      = AF_INET;
  sa_kaaddr.sin_port        = htons((U16)mgr->devconf->tcp_port_no);
  sa_kaaddr.sin_addr.s_addr = inet_addr((VP)mgr->devconf->tcp_ip_addr);

  flags = fcntl(cnfg->hdev.sock_id, F_GETFL, 0);
  fcntl(cnfg->hdev.sock_id, F_SETFL, flags | O_NONBLOCK); /* For making connect non blocking */
  while(1)
  {
    
    /* PRQA: Lint Message 64: suspicious ptr to ptr casting of sockaddr_in to sockaddr is intentional */
    /*lint -save -e64 */
    /* PRQA: Lint Message 826: suspicious ptr to ptr casting of sockaddr_in to sockaddr is intentional */
    /*lint -save -e826 */
    rc = connect(cnfg->hdev.sock_id, (VP)&sa_kaaddr, (S32)sizeof(sa_kaaddr));
    /*lint -restore */
    /*lint -restore */
    if(rc == E_OK)
    {
      /*agv2kor: CR-319*/
      /* clear communication err */
      TRACE_clear_comm_err(mgr);
      TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Established TCP/IP connection with Host\n");/*agv2kor*/
     /* Reset the flag connect_estab to default value when connection is established*/
      connect_estab=TRUE;
      cnfg->io_timeout = TRACE_SLEEP_SEC_10;      
      fcntl(cnfg->hdev.sock_id, F_SETFL, flags);
      break;
    }
    if(rc < E_OK) /*connect() returned error, because of error in connection or time out */
    {
#ifndef TRACE_DLT_EXIST

      if(TRUE == mgr->ctrl.cmd_info.rd_wait) /*Switching to trace_dlt_be request has come */
      {
        rc = E_OK;
#ifndef TRACE_ALD_INTERFACE_ENABLED
        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Switching to trace_dlt_be request\n");
#endif
        fcntl(cnfg->hdev.sock_id, F_SETFL, flags);
        break;
      }

#endif
      if( (ETIMEDOUT != errno) && (EINPROGRESS != errno) && (EALREADY != errno) && (ECONNREFUSED != errno) )/*Some actual error occured*/
      {
        /*Check the flag value when error and print the SYSLOG message only once */
        if (TRUE == connect_estab)
        {
           TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Real connect error from connect(), errno = %d\n", errno);
           connect_estab = FALSE;
        }
        rc = TRACE_TCPIP_cls(mgr, NULL);
        if(rc == E_OK)
        {
          rc = TRACE_TCPIP_opn(mgr, NULL);
          fcntl(cnfg->hdev.sock_id, F_SETFL, flags | O_NONBLOCK); /* For making connect non blocking */
        }
      }
      TRACE_sleep(TRACE_TCPIP_DLY);
    }
  }
  return rc;
}

/**
 * Trace read task epilog functionality.
 *
 * \parm mgr  Pointer to Trace manager
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_tcpip_rdt_epilog(TRACE_mgr* mgr)
{
  ER rc = TRACE_estab_connect(mgr);

  if(rc == E_OK)
  {
	  TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_WKUP_TCPCIP_WRT);
  }
  return rc;
}
/**
 * Trace read task epilog functionality.
 *
 * \parm mgr  Pointer to Trace manager
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_rdt_epilog(TRACE_mgr* mgr)
{
  ER rc = E_OK;

  switch(mgr->sh->trc.d_cnfg.actv_chan)
  {
    case TRACE_TCPIP:
      rc = TRACE_tcpip_rdt_epilog(mgr);
      break;

    case TRACE_PROXY:
    case TRACE_SIM:
    case TRACE_USBH:
    case TRACE_USBF:
    case TRACE_UART:
    case TRACE_NULLDEV:
    default:
      break;
  }
  return rc;
}

/**
 * Low-level RD for TCPIP
 *
 * \parm mgr  Pointer to Trace manager
 * \parm buf  User buffer to store read data
 * \parm rdsz Requested read size in bytes
 * \parm asiz Pointer to actual read size in bytes
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_TCPIP_rdext(TRACE_mgr* mgr, U8* buf, S32 rdsz, S32* asiz);
LOCAL ER TRACE_TCPIP_rdext(TRACE_mgr* mgr, U8* buf, S32 rdsz, S32* asiz)
{
  TRACE_io_cnfg* cnfg    = &mgr->io_cfg[TRACE_TCPIP];
  TRACE_TCPIP_ER rc      = E_OK;
  U8             err     = TRUE;
  fd_set rd_fds;
  struct timeval tm_probe;
  /* probe message does not comply to trace protocol
   * as this is only for probing purpose
   * however pktcnt, pktlength, procid, mst bytes are set to 0
   * to ensure backward compatibility[TTFis <= v3.1]
   * otherwise TTFis throws "invalid protocol" error
   */
   U8 probe[] = {0x01, 0x03, 0x00, 0x00};
   /* PRQA: QAC Message 3760: macros used from select.h produces the warnings */
   /* PRQA S 3760 L1 */
   /* PRQA: Lint Message 573: macros used from select.h produces the warnings */
   /*lint -save -e573 */
  FD_ZERO(&rd_fds);  
  FD_SET((S32)cnfg->hdev.sock_id, &rd_fds);
  tm_probe.tv_sec = cnfg->io_timeout;
  tm_probe.tv_usec = 0;
  
  /* use select() as it is more sensitive to errors than recv()
   * in order to recover from any abnormal link down (cable plug out)
   * it is required to have send() call. Otherwise recv/select() will
   * block for ever without able to noticing this error, especially when
   * no traces are sent(means no send() call)
   * Using select() with a 1 min timeout gives an oppurtunity to send 
   * probe message with which tcp stack detects link down error
   * and next recv() retuns with error
   */
  rc = select(cnfg->hdev.sock_id + 1, &rd_fds, NULL, NULL, &tm_probe);
  if(rc > E_OK)
  {
    if(FD_ISSET((S32)cnfg->hdev.sock_id, &rd_fds))
    {
      /*lint -restore */
	  /* PRQA L:L1 */
     
      /* socket is ready for reading */
      rc = recv(cnfg->hdev.sock_id, (VP)buf, (U32)rdsz, 0);

      /* Check if read was successfull */
      if(rc > E_OK)
      { 
	      *asiz = rc;
        err = FALSE;
      }      
    }
  }
  else
  { 
    err = FALSE; 

 #ifndef TRACE_DLT_EXIST
    if(TRUE != mgr->ctrl.cmd_info.rd_wait) /* Rd task is not asked to wait from switch task*/
    {
 #endif
    /* send connection probe only when 
     * there is no write between two consecutive writes
     * wr_idle == 0x01 : write is idle
     */
    if(mgr->ctrl.wr_idle != FALSE)
    {  
      if(TRUE == mgr->ctrl.tcp_keepAlive)/* If only tcp_keepAlive == 0x01,            /
                                           * Then only send keep alive packet sending */
      {
        if(send(cnfg->hdev.sock_id, (VP)probe, sizeof(probe), MSG_NOSIGNAL) == E_FAIL)
        {
          err = TRUE;
        }
      }
    }
    else
    {
      mgr->ctrl.wr_idle = TRUE;
    }
 #ifndef TRACE_DLT_EXIST
    }
 #endif
    
  }   
  if(err == TRUE)
  {    
    /* handle server diconnection 
     * select() returns success, but recv() returns 0
     */

    /* Diagnostic step to recover from socket disconnection */
    (void)shutdown(cnfg->hdev.sock_id, TRACE_SHUTDOWNOPTION);

    rc = TRACE_TCPIP_cls(mgr, NULL);
    if(rc == E_OK)
    {
      rc = TRACE_TCPIP_opn(mgr, NULL);
    }
    if(rc == E_OK)
    {
      rc = TRACE_estab_connect(mgr);
      TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_WKUP_TCPCIP_WRT);
    }
  }
  return rc;
}

/**
 * Read broken data packets over socket.
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_rd_pending_pkt(TRACE_mgr* mgr, U8* rawpkt, U8* buf, S32* asiz,
                              S32* indx, S32* rdlen, S32* tmplen, S32 sz);
LOCAL ER TRACE_rd_pending_pkt(TRACE_mgr* mgr, U8* rawpkt, U8* buf, S32* asiz,
                              S32* indx, S32* rdlen, S32* tmplen, S32 sz)
{
  ER  rc              = E_OK;
  S32 stage           = 1;
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_SIM];

  while((rc >= E_OK) && (stage < 3))
  {
    switch(stage)
    {
      case 1:
        if(mgr->sh->trc.d_cnfg.actv_chan == TRACE_SIM)
        {
	        *asiz = (S32)read(cnfg->hdev.dev_id, (VP)rawpkt, (U32)sz);
	        if(*asiz < 0)
	        {
            rc = E_FAIL;
	        }
	        else
	        {
            rc = *asiz;
	        }
        }
        else
        {
          if((g_TRACE_fnctbl.io_rd_ext[mgr->sh->trc.d_cnfg.actv_chan]) != NULL)
          {
            (g_TRACE_fnctbl.io_rd_ext[mgr->sh->trc.d_cnfg.actv_chan])(mgr,rawpkt, sz, asiz);
          }
        }       
        break;

      case 2:
        *tmplen += *asiz;
        if(*tmplen >= buf[TRACE_PKTSZ_FIELD])
        {
          memcpy((VP)(buf + (*rdlen)), (VP)rawpkt,
                 (U32)(buf[TRACE_PKTSZ_FIELD]) - (U32)(*rdlen));

          mgr->trc.in_msg = (TRACE_IN_MSG*)(VP) buf;
          rc = TRACE_process_cmd(mgr);
          if(rc >= E_OK)
          {
            *indx  = buf[TRACE_PKTSZ_FIELD] - (*rdlen);
            *asiz -= *indx;
            stage = 3;
          }
        }
        else
        {
          memcpy((VP)(buf + (*rdlen)), (VP)rawpkt, (U32)*asiz);
          *rdlen = *tmplen;
          stage = 0;
        }
        break;

        default:    
          break;
    }
    stage++;
  }
  return rc;
}

/**
 * Common read functionality for TCPIP, T-Sim & USBF
 *
 * \parm mgr    Pointer to Trace manager
 * \parm curq   Pointer to current packet in queue
 *
 * return \li E_OK     If successful
 *        \li E_IO     In case of socket errors
 *        \li E_DISWAI In case of     shutdown request
 *                     Else error returned by concrete implementation
 */
EXPORT ER TRACE_rd_common(TRACE_mgr* mgr, TRACE_q* curq);
EXPORT ER TRACE_rd_common(TRACE_mgr* mgr, TRACE_q* curq)
{
  U8  rawpkt[TRACE_MAX_FRAME_SZ] = {0};
  U8  buf[TRACE_MAX_FRAME_SZ]    = {0};
  ER  rc                         = E_OK;
  S32 asiz                       = 0;
  S32 indx                       = TRACE_PKTSZ_FIELD;
  S32 rdlen                      = 0;
  S32 tmplen                     = 0;
  S32 stage                      = 0;
  TRACE_io_cnfg* cnfg            = &mgr->io_cfg[TRACE_SIM];

/* PRQA: QAC Message 3201: All the cases would be executed based on the stage */
/* PRQA: QAC Message 3201: increment at the end of the switch statement           */
/* PRQA S 3201 L2 */
  while((rc >= E_OK) && (stage < 3))
  {
    switch(stage)
    {
      case 0:
        if(mgr->sh->trc.d_cnfg.actv_chan == TRACE_SIM)
        {
	  asiz = (S32)read(cnfg->hdev.dev_id,(VP)rawpkt, sizeof(rawpkt));
	  if(asiz > 0)
	  {
            rc = E_OK;
	  }
	  else
	  {
	    rc =(ER)asiz;
	  }
        }
        else
        {
          if((g_TRACE_fnctbl.io_rd_ext[mgr->sh->trc.d_cnfg.actv_chan]) != NULL)
          {
            (g_TRACE_fnctbl.io_rd_ext[mgr->sh->trc.d_cnfg.actv_chan])
                               (mgr,rawpkt, sizeof(rawpkt), &asiz);
          }
        }
        break;

      case 1:
        if(asiz > 0)
        {
          if((rawpkt[indx] < TRACE_MIN_PKT_SZ) ||
             (rawpkt[indx] > TRACE_MAX_PKT_SZ))
          {
            stage  = 3;
          }
        }
        else
        {
          stage = 3;
        }
        break;

      case 2:
        if(asiz >= rawpkt[indx])
        {
          mgr->trc.in_msg = (VP)&rawpkt[indx];

          (void)TRACE_process_cmd(mgr);
          if(asiz > rawpkt[indx])
          {
            asiz -= rawpkt[indx];
            indx += rawpkt[indx];
            stage = 0;
          }
        }
        else
        {
          memcpy((VP)buf, (VP)&rawpkt[indx], (U32)asiz);
          rdlen  = asiz;
          tmplen = asiz;
          rc = TRACE_rd_pending_pkt(mgr, rawpkt, buf, &asiz, &indx, &rdlen,
                                    &tmplen, sizeof(rawpkt));
          if(asiz > 0)
          {
            stage = 0;
          }
        }
        break;

        default:
          break;
    }
    stage++;
  }
/* PRQA L:L2 */    
  curq = curq;
  return rc;
}

/**
 * Read data from TCPIP socket
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK  On successful completion
 *        \li E_IO  On Socket error
 *                  Else error returned by concrete implementation
 */
LOCAL ER TRACE_TCPIP_rd(TRACE_mgr* mgr, TRACE_q* curq)
{
  return TRACE_rd_common(mgr, curq);
}

/**
 * Low-level WR for TCPIP
 *
 * \parm mgr  Pointer to Trace manager
 * \parm buf  User write buffer containing data
 * \parm rdsz Requested write size in bytes
 * \parm asiz Pointer to actual write size in bytes
 *
 * return \li E_OK     If successful
 *        \li E_IO     In case of socket errors
 *        \li E_DISWAI In case of shutdown request
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_TCPIP_wrext(TRACE_mgr* mgr, U8* buf, S32 wrsz, S32* asiz);
LOCAL ER TRACE_TCPIP_wrext(TRACE_mgr* mgr, U8* buf, S32 wrsz, S32* asiz)
{
  TRACE_io_cnfg* cnfg   = &mgr->io_cfg[TRACE_TCPIP];
  TRACE_TCPIP_ER rc     = E_OK;
 

  TRACE_flagAttr   Attr = {TRACE_TCPCIP_WAI_4_RDT_SIG, TWF_ORW | TWF_BITCLR, 
                                                        0, TMO_FEVR};

  while(1)
  {
    /* @kbm2kor: MSG_NOSIGNAL flag need to be set to ignore SIGPIPE signal 
     * when write on broken pipe is attempted. Modified for Linux impl.
     */
    rc = send(cnfg->hdev.sock_id, (VP)buf, (U32)wrsz, MSG_NOSIGNAL);

    if(rc > E_OK)
    {
      *asiz = rc;
      /* optimization for probing 
       * wr_idle == FALSE : write is active 
       */ 
      mgr->ctrl.wr_idle = FALSE;
      break;
    }
    else
    {
      /* Get connection state */
      rc = errno;
      if(rc == ENOBUFS)
      {
        TRACE_sleep(TRACE_SLEEP_MS_1); /* TRACE_SLEEP_MS_1 == 1 msec */
        continue;
      }
      else
      {
        /* raise communication err */
        TRACE_raise_comm_err(mgr);

        /* Diagnostic step to recover from socket disconnection */
        (void)shutdown(cnfg->hdev.sock_id, TRACE_SHUTDOWNOPTION);

        /* Wait for signal from RD task */
        (void)TRACE_wait_flag(&mgr->sh->evt_id[EVT_TRC_Q], &Attr);

        /* clear communication err */
        TRACE_clear_comm_err(mgr);
        rc = E_OK;
      }
      break;
    }
  }
  return rc;
}

/**
* Common write functionality for TCPIP and T-Sim
*
* \parm mgr    Pointer to Trace manager
* \parm curq   Pointer to current packet in queue
* \parm rawpkt Pointer to local buffer for framing packet
* \parm wrsz   Pointer to a variable holding bytes written info
*
* return \li E_OK     If successful
*        \li E_IO     In case of socket errors
*        \li E_DISWAI In case of shutdown request
*                     Else error returned by concrete implementation
*/
EXPORT void TRACE_wr_common(TRACE_mgr* mgr, TRACE_q* curq, U8* rawpkt, S32* wrsz);
EXPORT void TRACE_wr_common(TRACE_mgr* mgr, TRACE_q* curq, U8* rawpkt, S32* wrsz)
{
    TRACE_TRACEOUT_MSG* pkt         = NULL;
    TRACE_OUT_PKT*      trace_pkt   = NULL;

    pkt         = (TRACE_TRACEOUT_MSG*)(VP) curq->buf;
    trace_pkt   = (VP) &pkt->pkt;

        
    /* dhd3kor<12.10.2010>:Fix for SWGII-3341 
     * Packet count is not updated when the frame is constructed for
     * Timestamp & trace message.
     */
    rawpkt[TRACE_FRAME_CNT] = 0;
    if(pkt->tm_stamp != TRACE_TIMESTAMP_DISABLE)
    {
      rawpkt[TRACE_FRAME_CNT]++;
      TRACE_construct_frame(trace_pkt, rawpkt + (*wrsz), wrsz);
    }

    trace_pkt = (TRACE_OUT_PKT*)(VP) &pkt->pkt.trc_msg.out_pkt;
    TRACE_construct_frame(trace_pkt, rawpkt + (*wrsz), wrsz);
    rawpkt[TRACE_FRAME_CNT]++;

    mgr = mgr;    
}

/**
 * Write data to TCPIP socket
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to current packet
 *
 * return \li E_OK     If successful
 *        \li E_IO     In case of socket errors
 *        \li E_DISWAI In case of shutdown request
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_TCPIP_wr(TRACE_mgr* mgr, TRACE_q* curq)
{
  S32   asize   = 0;
  ER    rc      = E_OK;
  S32   wrsz    = 1;
  U8  rawpkt[sizeof(TRACE_TRACEOUT_MSG) + 2];

  TRACE_wr_common(mgr, curq, rawpkt, &wrsz);

  if(wrsz > 0)
  {
    rc = TRACE_TCPIP_wrext(mgr, (VP)rawpkt, wrsz, &asize);
  }
  return rc;
}

/**
 * Transmit packet via UART channel
 *
 * \parm mgr      Pointer to Trace manager
 * \parm buf      User buffer
 * \parm sz       Size of user buffer
 * \parm asize    Pointer to hold info about actual number of bytes
 *                successfully transmitted via UART
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_uart_wr_subfunc(TRACE_mgr* mgr, S8* buf, S32 sz, S32* asize);
LOCAL ER TRACE_uart_wr_subfunc(TRACE_mgr* mgr, S8* buf, S32 sz, S32* asize)
{
  TRACE_io_cnfg* cnfg    = &mgr->io_cfg[TRACE_UART];
  U32  i                 = 0;
  ER  rc                 = E_OK;
  
  TRACE_flagAttr   Attr  = {TRACE_ACK_WAIT_WRT | TRACE_NACK_WAIT_WRT, TWF_ORW | TWF_BITCLR, 
                              0, cnfg->io_dev.dev_uart.rx_timeout};

  if(sz == (S32)1)
  {
    *asize = (ER)write(cnfg->hdev.dev_id, buf, (U32)sz);
    if(*asize < 0)
    {
        rc = (ER)*asize;
    }
  }
  else
  {
    Attr.flg_ptn = TRACE_NACK_WAIT_WRT;
    do
    {
      if((TRACE_NACK_WAIT_WRT == (Attr.flg_ptn & TRACE_NACK_WAIT_WRT)) &&
         (ETIMEDOUT != rc))
      {	      
	*asize = write(cnfg->hdev.dev_id, buf, (U32)sz);
	if(*asize < 0)
        {
            rc = (ER)*asize;
	      }
	      else
	      {
	          rc = (ER)E_OK;
	      }
      }
      if(rc >= E_OK)
      {
	      TRACE_wait_flag(&mgr->sh->evt_id[EVT_TRC_Q], &Attr);
      }
      if((Attr.flg_ptn & TRACE_ACK_WAIT_WRT) == TRACE_ACK_WAIT_WRT)
      {
        i++;
      }
    }while((i == 0) && (rc == E_OK));
  }
  if(ETIMEDOUT == rc)
  {
    rc = E_OK;
    TRACE_raise_comm_err(mgr);/*agv2kor CR 319*/
  }
  return rc;
}

/**
 * Dummy function
 *
 * \parm mgr  Pointer to Trace manager
 * \parm buf  Pointer to packet to be written to IO device
 * \parm rdsz size to be read/write
 * \parm asiz pointer to actual size read/write

 * return \li E_FAIL Non supported functionality
 */
LOCAL ER TRACE_ioext_null(TRACE_mgr* mgr, U8* buf, S32 rdsz, S32* asiz)
{
  ER rc = E_FAIL;

  /* to do */

  mgr = mgr;
  buf = buf;
  asiz = asiz;
  rdsz = rdsz;
  return rc;
}


/**
 * Dummy function
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to packet to be written to IO device
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_dev_null(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */

  mgr = mgr;
  curq = curq;

  return rc;
}

/**
 * Open UART channel
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to packet to be written to IO device
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_UART_opn(TRACE_mgr* mgr, TRACE_q* curq)
{
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_UART];
  TRACE_uart*    dev  = (VP)&cnfg->io_dev;
  ER             rc   = E_OK;
  S32            asiz = 0;

  if(cnfg->hdev.dev_id == 0)
  {
    rc = open((char*)dev->nm, O_RDWR |  O_NOCTTY | O_NDELAY | O_SYNC | O_NONBLOCK);
    if(rc > E_OK)
    {
      struct termios rsmode;
      fcntl(rc, F_SETFL, 0);/*for normal blocking behaviour*/
      cnfg->hdev.dev_id = rc;
      tcgetattr(cnfg->hdev.dev_id, &rsmode);
      
      /*API for making data mode as RAW*/
      cfmakeraw(&rsmode);  
         /* Input flags - Turn off input processing
       convert break to null byte, no CR to NL translation,
       no NL to CR translation, don't mark parity errors or breaks
       no input parity check, don't strip high bit off,
       no XON/XOFF software flow control
    */
    rsmode.c_iflag &= ~(IGNBRK | BRKINT | ICRNL |
                        INLCR | PARMRK | INPCK | ISTRIP | IXON);

    /* Output flags - Turn off output processing
       no CR to NL translation, no NL to CR-NL translation,
       no NL to CR translation, no column 0 CR suppression,
       no Ctrl-D suppression, no fill characters, no case mapping,
       no local output processing

       config.c_oflag &= ~(OCRNL | ONLCR | ONLRET |
                           ONOCR | ONOEOT| OFILL | OLCUC | OPOST);
    */
    rsmode.c_oflag = 0;

    /* No line processing:
       echo off, echo newline off, canonical mode off,
       extended input processing off, signal chars off
    */
    rsmode.c_lflag &= ~(ECHO | ECHONL | ICANON | IEXTEN | ISIG);

    /* Turn off character processing
       clear current char size mask, no parity checking,
       no output processing, force 8 bit input
    */
    rsmode.c_cflag &= ~(CSIZE | PARENB);
    rsmode.c_cflag |= CS8;

    /* One input byte is enough to return from read()
       Inter-character timer off
    */
    rsmode.c_cc[VMIN]  = 1;
    rsmode.c_cc[VTIME] = 0;
 
      /*Set baud rate as 115200*/
      cfsetispeed(&rsmode, B115200);
      cfsetospeed(&rsmode, B115200);

      /*Set the new options for the port...        */
      tcsetattr(cnfg->hdev.dev_id, TCSANOW, &rsmode);
      TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Opened UART connection with Host\n");/*agv2kor*/
      asiz = asiz; /*To suppress warning*/
    }

  }
  curq = curq;
  return rc;
}

/**
 * Close UART channel
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to packet to be written to IO device
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */

LOCAL ER TRACE_UART_cls(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER             rc   = E_OK;
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_UART];

  if(cnfg->hdev.dev_id > 0)
  {
    rc = close(cnfg->hdev.dev_id);
    if(rc == E_OK)
    {
      cnfg->hdev.dev_id = 0;
    }
  }
  curq = curq;
  return rc;
}

/**
 * Handle ACK, NACK, RESET, SYNCH messages from TTFis
 *
 * \parm mgr Pointer to Trace manager
 * \parm cmd Command received from TTFis
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL void TRACE_spcl_commands(TRACE_mgr* mgr, U8 cmd);
LOCAL void TRACE_spcl_commands(TRACE_mgr* mgr, U8 cmd)
{
  switch(cmd)
  {
    case TRACE_RESET:
    case TRACE_ACK:
	    TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_ACK_SND_WRT);
      break;

    case TRACE_SYNC: /* No action required */
      break;

    case TRACE_NACK:
    default:
      TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_NACK_SND_WRT);
    break;
  }
}

/**
 * Get single payload packet from IO device
 *
 * \parm mgr    Pointer to Trace manager
 * \parm rawpkt Pointer to buffer to hold incoming packet
 * \parm sz     Pointer to variable which holds received byte count
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_get_single_payld(TRACE_mgr* mgr, U8* rawpkt, S32* sz);
LOCAL ER TRACE_get_single_payld(TRACE_mgr* mgr, U8* rawpkt, S32* sz)
{
  TRACE_io_cnfg* cnfg      = &mgr->io_cfg[TRACE_UART];
  ER             rc        = E_OK;
  S32            asize     = 1;
  S32            prevAsize = 0;

  /* Read packet length */
  asize = (S32)read(cnfg->hdev.dev_id, (VP)&rawpkt[TRACE_LEN_FIELD], TRACE_RD_1_BYTE);

  if(asize > E_OK)
  {
    asize = 0;
    do{
       prevAsize = (S32)read(cnfg->hdev.dev_id, (VP)&rawpkt[TRACE_DAT+asize], (rawpkt[TRACE_LEN_FIELD] - (U8)1) - asize);
       asize += prevAsize;
     }while(asize < (rawpkt[TRACE_LEN_FIELD]-1) && (asize > 0) ); /* Fix for UART communication is not working in Gen3*/
  }
  if(asize < E_OK)
  {
      rc = E_FAIL;
  }
  else
  {
      rc = E_OK;
  }

  if(rc >= E_OK)
  {
    /* Read remaining packet */
  }

  if(rc >= E_OK)
  {
    *sz += rawpkt[TRACE_LEN_FIELD];
  }
  return rc;
}

/**
 * Process incomming frames
 *
 * \parm mgr    Pointer to Trace manager
 * \parm frmcnt Number of frames
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_process_frames(TRACE_mgr* mgr, U8 frmcnt);
LOCAL ER TRACE_process_frames(TRACE_mgr* mgr, U8 frmcnt)
{
  TRACE_io_cnfg* cnfg   = &mgr->io_cfg[TRACE_UART];
  U8            rawpkt[TRACE_MAX_FRAME_SZ + 1]  = {0};
  S32           stage                           = 0;
  ER            rc                              = E_OK;
  S32           i                               = 0;
  U32           sz                              = 1;
  U8            calc_cs                         = 0;
  U8            org_cs                          = 0;
  S32           asize                           = 0;

  TRACE_PROCESS_FRAMES frame                    = {NULL};

  frame.pkt = &rawpkt[1];

  while((stage < 3) && (rc >= E_OK))
  {
    switch(stage)
    {
      case 0: /* Assemble all frames */
        rawpkt[0] = frmcnt;
        for(; ((i < frmcnt) && (rc >= E_OK) && (sz < sizeof(rawpkt))); i++)
        {
          rc = TRACE_get_single_payld(mgr, &rawpkt[sz], (VP)&sz);
        }
        if(rc >= E_OK)
        {
          TRACE_calculate_checksum(rawpkt, sz,
                                   &calc_cs, TRACE_RX_CHECKSUM);
        }
        break;

      case 1:  /* Get checksum byte */
	      asize = (S32)read(cnfg->hdev.dev_id, (VP)&org_cs, TRACE_RD_1_BYTE);
        if(asize < 0)
	      {
          rc = (ER)asize;
	      }
	      else
	      {
	        rc = E_OK;
	      }
	break;

      case 2: /* Verify checksum */
        if(org_cs == calc_cs)
        {
          TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_ACK_SND_WRT);

          for(i = 0; ((i < frmcnt) && (rc >= E_OK)); i++)
          {
            mgr->trc.in_msg = frame.msg;
            rc = TRACE_process_cmd(mgr);
            frame.pkt = frame.pkt + (*frame.pkt);
          }
        }
        else
        {
          TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_NACK_SND_WRT);
        }
        break;

        default:
          break;
    }
    stage++;
  }
  if(rc < E_OK)
  {
    TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_NACK_SND_WRT);
  }
  return rc;
}

/**
 * Read data from UART channel
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to packet to be processed
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_UART_rd(TRACE_mgr* mgr, TRACE_q* curq)
{
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_UART];
  ER  rc              = E_OK;
  S32 asize           = 1;
  U8  frmcnt          = 0;

  /* Read frame count */
  asize = (S32)read(cnfg->hdev.dev_id, (VP)&frmcnt, TRACE_RD_1_BYTE);
  if(asize < 0)
  {
      rc = (ER)asize;
  }
  else
  {
      rc = E_OK;
  }

  if(rc >= E_OK)
  {
    TRACE_clear_comm_err(mgr);/*CR-319, agv2kor*/
    /*To clear the comm err raised due to time out*/

    if((frmcnt == TRACE_ACK)   || (frmcnt == TRACE_NACK) ||
       (frmcnt == TRACE_RESET) || (frmcnt == TRACE_SYNC))
    {
      TRACE_spcl_commands(mgr, frmcnt);
    }
    else if((frmcnt >= TRACE_MIN_FRAME_CNT) && (frmcnt <= TRACE_MAX_FRAME_CNT))
    {
      rc = TRACE_process_frames(mgr, frmcnt);
    }
    else { }
  }
  curq = curq;
  return rc;
}

/**
 * Write data to UART channel
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to packet to be processed
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
LOCAL ER TRACE_UART_wr(TRACE_mgr* mgr, TRACE_q* curq)
{
  U8                  ack_nack         = 0;
  S32                 asize            = 0;
  ER                  rc               = E_OK;
  static S32          wrsz             = 1;
  TRACE_TRACEOUT_MSG* pkt              = NULL;
  TRACE_OUT_PKT*      trace_pkt        = NULL;
  static U8 rawpkt[TRACE_MAX_FRAME_SZ] = {0};
  TRACE_UART_WRI      tmp              = {NULL};
  BOOL              copy_pending       = TRUE;

  if(mgr->flow_ctrl != TRACE_NO_SIG)
  {
    ack_nack = (U8)mgr->flow_ctrl;
    mgr->flow_ctrl = TRACE_NO_SIG;
    wrsz = 1;
    rawpkt[0]  = ack_nack;
    rc = TRACE_uart_wr_subfunc(mgr, (VP)rawpkt, wrsz, &asize);
  }
  else
  {
    tmp.buf = curq->buf;
    pkt       = tmp.omsg;
    trace_pkt = (TRACE_OUT_PKT*) (VP) &pkt->pkt;

    while(copy_pending == TRUE)
    {
      if(((size_t)wrsz + sizeof(TRACE_TRACEOUT_MSG))  < (sizeof(rawpkt)- 1))
      {
        /* if no more pkts, write to IO */
        if(curq->sh->remain == 0)
        {
          if(pkt->tm_stamp != TRACE_TIMESTAMP_DISABLE)
          {
            rawpkt[TRACE_FRAME_CNT]++;
            TRACE_construct_frame(trace_pkt, &rawpkt[wrsz], &wrsz);
          }
          trace_pkt = (TRACE_OUT_PKT*)(VP) &pkt->pkt.trc_msg.out_pkt;
          TRACE_construct_frame(trace_pkt, &rawpkt[wrsz], &wrsz);
          rawpkt[TRACE_FRAME_CNT]++;

          
          TRACE_calculate_checksum(rawpkt, (U32)wrsz, &rawpkt[wrsz],TRACE_TX_CHECKSUM);
          
          /* write to IO */
          rc = TRACE_uart_wr_subfunc(mgr, (VP)rawpkt, (wrsz+1), &asize);
          wrsz = 1;
          rawpkt[TRACE_FRAME_CNT] = 0;
        }
        else
        {
          /* keep assembling pkts into frame */
          if(pkt->tm_stamp != TRACE_TIMESTAMP_DISABLE)
          {
            rawpkt[TRACE_FRAME_CNT]++;
            TRACE_construct_frame(trace_pkt, &rawpkt[wrsz], &wrsz);
          }
          trace_pkt = (TRACE_OUT_PKT*)(VP) &pkt->pkt.trc_msg.out_pkt;
          TRACE_construct_frame(trace_pkt, &rawpkt[wrsz], &wrsz);
          rawpkt[TRACE_FRAME_CNT]++;
        }
        copy_pending = FALSE;
      }
      else
      {
        /* Frame is full */
        TRACE_calculate_checksum(rawpkt, (U32)wrsz, &rawpkt[wrsz],TRACE_TX_CHECKSUM);
        rc = TRACE_uart_wr_subfunc(mgr, (VP)rawpkt, (wrsz+1), &asize);
        wrsz = 1;
        rawpkt[TRACE_FRAME_CNT] = 0;
      }
    }
  }

  return rc;
}


/**
 * Open USBH device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_USBH_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_opn(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */

  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Open USBH device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_USBH_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_cls(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */

  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Read data from USBH device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_USBH_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_rd(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */

  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Read data to USBH device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_USBH_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_USBH_wr(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */

  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Open T-Sim IO device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK     On successful completion
 *        \li E_DISWAI On shutdown request
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_SIM_opn(TRACE_mgr* mgr, TRACE_q* curq)
{
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_SIM];
  TRACE_sim*     dev  = (VP)&cnfg->io_dev;
  ER             rc   = E_OK;

  if(cnfg->hdev.dev_id == 0)
  {
    rc = open((char*)dev->nm, O_RDWR);
    if(rc > E_OK)
    {
      cnfg->hdev.dev_id = rc;
    }
  }
  curq = curq;
  return rc;
}

/**
 * Close T-Sim IO device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK     On successful completion
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_SIM_cls(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER             rc   = E_OK;
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_SIM];

  if(cnfg->hdev.dev_id > 0)
  {
    rc = close(cnfg->hdev.dev_id);
    if(rc == E_OK)
    {
      cnfg->hdev.dev_id = 0;
    }
  }
  curq = curq;

  return rc;
}

/**
 * Read data from T-Sim IO device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_OK     On successful completion
 *        \li E_DISWAI On shutdown request
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_SIM_rd(TRACE_mgr* mgr, TRACE_q* curq)
{
  return TRACE_rd_common(mgr, curq);
}

/**
 * Write data to T-Sim IO device
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to active packet
 *
 * return \li E_OK     On successful completion
 *        \li E_DISWAI On shutdown request
 *                     Else error returned by concrete implementation
 */
LOCAL ER TRACE_SIM_wr(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER  rc      = E_OK;
  S32   wrsz  = 1;
  U8  rawpkt[sizeof(TRACE_TRACEOUT_MSG) + 2];
  TRACE_io_cnfg* cnfg = &mgr->io_cfg[TRACE_SIM];

  TRACE_wr_common(mgr, curq, rawpkt, &wrsz);

  if(wrsz > 0)
  {
    rc = write(cnfg->hdev.dev_id, (VP)rawpkt, (U32)wrsz);
  }
  return rc;
}

/**
 * Open Trace proxy
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_PROXY_opn(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_opn(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */
  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Close Trace proxy
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_PROXY_cls(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_cls(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */
  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Read data from Trace Proxy
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Not used
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_PROXY_rd(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_rd(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;
   
  /* to do */
  mgr = mgr;
  curq = curq;
  return rc;
}

/**
 * Write data to Trace Proxy
 *
 * \parm mgr  Pointer to Trace manager
 * \parm curq Pointer to current packet
 *
 * return \li E_NOSPT Non supported functionality
 */
LOCAL ER TRACE_PROXY_wr(TRACE_mgr* mgr, TRACE_q* curq);
LOCAL ER TRACE_PROXY_wr(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER rc = E_FAIL;

  /* to do */
  mgr = mgr;
  curq = curq;
  return rc;
}


